///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/**
 * \file CylinderObject.h
 * \brief Contains the definition of the StdObjects::CylinderObject class.
 */

#ifndef __STDOBJ_CYLINDER_OBJECT_H
#define __STDOBJ_CYLINDER_OBJECT_H

#include <core/Core.h>
#include <core/scene/objects/geometry/SimpleGeometryObject.h>
#include <core/scene/animation/controller/Controller.h>
#include <core/gui/properties/PropertiesEditor.h>

#include <stdobjects/StdObjects.h>

namespace StdObjects {

/**
 * \brief The procedural cylinder geometry object.
 *
 * \author Alexander Stukowski
 */
class STDOBJECTS_DLLEXPORT CylinderObject : public SimpleGeometryObject
{
public:
	/// \brief Default constructors.
	/// \param isLoading Specifies whether the object's data fields will be initialized from the
	///                  data stored in a scene file after the instance has been created.
	CylinderObject(bool isLoading = false);

	/// \brief Returns the controller that controls the radius of the cylinder object.
	/// \return The internal controller object.
	/// \sa setRadiusController()
	FloatController* radiusController() const { return _radius; }

	/// \brief Returns the controller that controls the height of the cylinder object.
	/// \return The internal controller object.
	/// \sa setHeightController()
	FloatController* heightController() const { return _height; }

	/// \brief Replaces the controller that controls the radius of the cylinder object with a new controller.
	/// \param ctrl The new controller for the radius parameter.
	/// \undoable
	/// \sa radiusController()
	void setRadiusController(FloatController* ctrl) { _radius = ctrl; }

	/// \brief Replaces the controller that controls the height of the cylinder object with a new controller.
	/// \param ctrl The new controller for the height parameter.
	/// \undoable
	/// \sa heightController()
	void setHeightController(FloatController* ctrl) { _height = ctrl; }

	// From SceneObject:

	/// \brief Asks the object for its validity interval at the given time.
	/// \param time The animation time at which the validity interval should be computed.
	/// \return The maximum time interval that contains \a time and during which the object does not change.
	virtual TimeInterval objectValidity(TimeTicks time);

protected:

	// From SimpleGeometryObject:

	/// \brief Builds the mesh representation of this geometry object.
	/// \param[in] time The animation time at which to generate the mesh.
	/// \param[out] mesh The mesh object that is be filled by the method.
	/// \param[out] meshValidity The method stores the validty interval of the generated mesh in this output variable.
	virtual void buildMesh(TimeTicks time, TriMesh& mesh, TimeInterval& meshValidity);

protected:

	/// This controller stores the radius of the cylinder.
	ReferenceField<FloatController> _radius;

	/// This controller stores the height of the cylinder.
	ReferenceField<FloatController> _height;

	/// This stores the number radius segments.
	PropertyField<int> _radiusSegments;

	/// This stores the number height segments.
	PropertyField<int> _heightSegments;

	/// Controls the smoothing of the cylinder faces
	PropertyField<bool> _smoothFaces;

private:

	Q_OBJECT
	DECLARE_SERIALIZABLE_PLUGIN_CLASS(CylinderObject)
	DECLARE_REFERENCE_FIELD(_radius)
	DECLARE_REFERENCE_FIELD(_height)
	DECLARE_PROPERTY_FIELD(_radiusSegments)
	DECLARE_PROPERTY_FIELD(_heightSegments)
	DECLARE_PROPERTY_FIELD(_smoothFaces)
};


/**
 * \brief A properties editor for the CylinderObject class.
 *
 * This editor class creates and manages the user interface through which the
 * user can alter the object's parameters.
 *
 * \author Alexander Stukowski
 */
class CylinderObjectEditor : public PropertiesEditor
{
protected:

	/// Creates the user interface controls for the editor.
	virtual void createUI(const RolloutInsertionParameters& rolloutParams);

private:

	Q_OBJECT
	DECLARE_PLUGIN_CLASS(CylinderObjectEditor)
};

};

#endif // __STDOBJ_CYLINDER_OBJECT_H
