/*  This file is part of SAIL (https://github.com/HappySeaFox/sail)

    Copyright (c) 2021 Dmitry Baryshev

    The MIT License

    Permission is hereby granted, free of charge, to any person obtaining a copy
    of this software and associated documentation files (the "Software"), to deal
    in the Software without restriction, including without limitation the rights
    to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
    copies of the Software, and to permit persons to whom the Software is
    furnished to do so, subject to the following conditions:

    The above copyright notice and this permission notice shall be included in all
    copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
    AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
    LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
    OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
    SOFTWARE.
*/

#include <stdbool.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>

#include <sail-common/sail-common.h>

#include "helpers.h"

/*
 * Codec-specific state.
 */
struct wal_state {
    struct sail_io *io;
    const struct sail_load_options *load_options;
    const struct sail_save_options *save_options;

    unsigned frame_number;

    struct WalFileHeader wal_header;
    unsigned width;
    unsigned height;
};

static sail_status_t alloc_wal_state(struct sail_io *io,
                                        const struct sail_load_options *load_options,
                                        const struct sail_save_options *save_options,
                                        struct wal_state **wal_state) {

    void *ptr;
    SAIL_TRY(sail_malloc(sizeof(struct wal_state), &ptr));
    *wal_state = ptr;

    **wal_state = (struct wal_state) {
        .io           = io,
        .load_options = load_options,
        .save_options = save_options,

        .frame_number  = 0,
        .width         = 0,
        .height        = 0,
    };

    return SAIL_OK;
}

static void destroy_wal_state(struct wal_state *wal_state) {

    if (wal_state == NULL) {
        return;
    }

    sail_free(wal_state);
}

/*
 * Decoding functions.
 */

SAIL_EXPORT sail_status_t sail_codec_load_init_v8_wal(struct sail_io *io, const struct sail_load_options *load_options, void **state) {

    *state = NULL;

    /* Allocate a new state. */
    struct wal_state *wal_state;
    SAIL_TRY(alloc_wal_state(io, load_options, NULL, &wal_state));
    *state = wal_state;

    /* Read WAL header. */
    SAIL_TRY(wal_private_read_file_header(wal_state->io, &wal_state->wal_header));

    wal_state->width = wal_state->wal_header.width;
    wal_state->height = wal_state->wal_header.height;

    return SAIL_OK;
}

SAIL_EXPORT sail_status_t sail_codec_load_seek_next_frame_v8_wal(void *state, struct sail_image **image) {

    struct wal_state *wal_state = state;

    if (wal_state->frame_number == 4) {
        SAIL_LOG_AND_RETURN(SAIL_ERROR_NO_MORE_FRAMES);
    }

    if (wal_state->frame_number > 0) {
        wal_state->width /= 2;
        wal_state->height /= 2;
    }

    struct sail_image *image_local;
    SAIL_TRY(sail_alloc_image(&image_local));

    if (wal_state->load_options->options & SAIL_OPTION_SOURCE_IMAGE) {
        SAIL_TRY_OR_CLEANUP(sail_alloc_source_image(&image_local->source_image),
                            /* cleanup */ sail_destroy_image(image_local));

        image_local->source_image->pixel_format = SAIL_PIXEL_FORMAT_BPP8_INDEXED;
        image_local->source_image->compression  = SAIL_COMPRESSION_NONE;
    }

    image_local->width          = wal_state->width;
    image_local->height         = wal_state->height;
    image_local->pixel_format   = SAIL_PIXEL_FORMAT_BPP8_INDEXED;
    image_local->bytes_per_line = sail_bytes_per_line(image_local->width, image_local->pixel_format);

    SAIL_TRY_OR_CLEANUP(wal_private_assign_palette(image_local),
                        /* cleanup */ sail_destroy_image(image_local));
    SAIL_TRY_OR_CLEANUP(wal_private_assign_meta_data(&wal_state->wal_header, &image_local->meta_data_node),
                        /* cleanup */ sail_destroy_image(image_local));

    SAIL_TRY_OR_CLEANUP(wal_state->io->seek(wal_state->io->stream, wal_state->wal_header.offset[wal_state->frame_number], SEEK_SET),
                        /* cleanup */ sail_destroy_image(image_local));

    wal_state->frame_number++;

    *image = image_local;

    return SAIL_OK;
}

SAIL_EXPORT sail_status_t sail_codec_load_frame_v8_wal(void *state, struct sail_image *image) {

    struct wal_state *wal_state = state;

    SAIL_TRY(wal_state->io->strict_read(wal_state->io->stream, image->pixels, (size_t)image->bytes_per_line * image->height));

    return SAIL_OK;
}

SAIL_EXPORT sail_status_t sail_codec_load_finish_v8_wal(void **state) {

    struct wal_state *wal_state = *state;

    *state = NULL;

    destroy_wal_state(wal_state);

    return SAIL_OK;
}

/*
 * Encoding functions.
 */

SAIL_EXPORT sail_status_t sail_codec_save_init_v8_wal(struct sail_io *io, const struct sail_save_options *save_options, void **state) {

    (void)io;
    (void)save_options;
    (void)state;

    SAIL_LOG_AND_RETURN(SAIL_ERROR_NOT_IMPLEMENTED);
}

SAIL_EXPORT sail_status_t sail_codec_save_seek_next_frame_v8_wal(void *state, const struct sail_image *image) {

    (void)state;
    (void)image;

    SAIL_LOG_AND_RETURN(SAIL_ERROR_NOT_IMPLEMENTED);
}

SAIL_EXPORT sail_status_t sail_codec_save_frame_v8_wal(void *state, const struct sail_image *image) {

    (void)state;
    (void)image;

    SAIL_LOG_AND_RETURN(SAIL_ERROR_NOT_IMPLEMENTED);
}

SAIL_EXPORT sail_status_t sail_codec_save_finish_v8_wal(void **state) {

    (void)state;

    SAIL_LOG_AND_RETURN(SAIL_ERROR_NOT_IMPLEMENTED);
}
